****************************************
* Fluke 87 Data Acquisition sourcecode *
*                                      *
*        Author: Derek Matsunaga       *
*      Revision: 2.2                   *
* Revision date: October 3, 1993       *
*     Processor: 68HC705K1             *
*                                      *
* Revision 2.2 corrects serial timing. *
****************************************

* Set-up 68HC05K1 equate table.

porta   equ     $0000   ;Port a address.
portb   equ     $0001   ;Port b address.
ddra    equ     $0004   ;Data direction register a.
ddrb    equ     $0005   ;Data direction register b.

* Define RAM locations.

        org     $00e0   ;The beginning of RAM.
temp1   equ     $00e0   ;A general purpose temporary register.
lcount  equ     $00e1   ;The register which keeps track of low time.
lowS    equ     $00e1   ;The low "S" register from the meter.
hcount  equ     $00e2   ;The register which keeps track of high time.
highS   equ     $00e2   ;The high "S" register from the meter.
scount  equ     $00e3   ;The "S" register counter.
bitcount        equ     $00e3   ;The number of bits per byte received.
bytecount       equ     $00e4   ;The number of 8 bit bytes received (#18 max).
firstbyte       equ     $00e5   ;The location of the first byte in the record.
string  equ     $00e5   ;The start of the output string.
checksum        equ     $00f8   ;The checksum register.


* Define vector table.

        org     vectors
        fdb     rom
        fdb     rom
        fdb     rom
        fdb     reset

* Initialize ports and registers.

        org     $0200   ;The beginning of program ROM.
reset   lda     #$ff
        sta     ddra    ;Configure all porta pins as outputs.
        sta     porta   ;Write all ones to porta.
        clr     ddrb    ;Configure both portb pins as inputs.
        clr     bytecount       ;Initialize counting registers.
        clr     bitcount
        clr     checksum        ;Clear the checksum register.

* Since each nibble begins with a "1" as a start bit, the deadtime during
* valid data transmission will not exceed the 7.8ms nibble time
* (1.3ms/bit * 6 bits, see explanation diagram).  So, a complete record of
* data has just been sent if deadtime exceeds the 7.8ms nibble time.  Use
* about 11ms of dead time to prevent false record triggering (for noise immunity).

* Look for about 11ms of deadtime before reading data.  Due to the length of the
* deadtime, a 16 bit counter is needed.  The "temp1" location is used to store the
* "high" byte (upper 8 bits) of the 16 bit counter.  The X register is used to store the "low"
* byte.  See sourcecode validation for detailed timing information.

start   clrx            ;Clear the X register.
        stx     temp1   ;Clear the "high" byte of X.

loop    brset   0,portb,start   ;Loop & clear X until portb0 goes low.

        decx            ;Decrement the X register.
        bne     loop    ;If X>0, go to top of loop.

        inc     temp1   ;Directly increment the high byte of the counter.
        lda     temp1   ;Load the new high byte into the accumulator.
        cmp     #$08    ;Temp1 will be 8 when about 11.3ms of
                        ;   deadtime has occurred.
        bne     loop

* About 11.3ms of deadtime has been detected at portb0.  We are now ready
* to start reading the data from the meter.  The beginning of the actual
* data occurs with a low to high transition following the deadtime.

* Wait until deadtime is over (a high is detected at portb0).
* Each nibble is started with a 1.3ms high (a startbit).

getstart        brclr   0,portb,getstart        ;Read portb0 until it goes high.

* When the high at portb0 is detected, initialize the timer locations.

nextbit ldx     #$B9    ;Initialize the X register with the bit time.
        clr     lcount  ;Initialize the low count register.
        clr     hcount  ;Initialize the high count register.

timer   decx            ;Decrement X register to keep track of time.
        beq     timeout ;Check portb until X=0 (the timer has expired).

chkport brset   0,portb,high    ;Check for a high at portb0.

low     inc     lcount  ;Increment the low count location.
        bra     timer   ;Go back to timer.

high    inc     hcount  ;Increment the high count location.
        bra     timer   ;Go back to timer.

timeout lda     bitcount        ;Load the bit counter.
        cmp     #$00    ;Determine if it is the start bit of the 1st nibble.
        beq     housekeep       ;Ignore this bit if it is the start bit.
        cmp     #$05    ;Determine if it is the start bit of the 2nd nibble.
        beq     housekeep       ;Ignore this bit if it is the start bit.

        lda     lcount  ;Load the number of low counts.
        sub     hcount  ;Subtract the number of high counts.
                        ;C will be set if there were more highs than lows.
        ldx     bytecount       ;Load the record pointer into X.
        rol     firstbyte,x     ;Roll the carry bit onto the current record byte.

housekeep       inc     bitcount        ;Increment the number of bits read.
        lda     bitcount        ;Load the incremented value into A.
        cmp     #$05    ;If bitcount is #5, get ready for next nibble.
        beq     add_chksum
        cmp     #$0a    ;If bitcount is #10, get ready for next byte.
        bne     nextbit

add_chksum      lda     firstbyte,x
        and     #$0f
        add     checksum
        sta     checksum

        lda     bitcount
        cmp     #$05
        beq     getstart

nextbyte        inc     bytecount       ;Increment the number of bytes read.
        lda     bytecount       ;Load the incremented number into A.
        cmp     #$12    ;Determine if #18 bytes have been read.
        beq     cs_test ;Start processing data if #18 bytes have been read.
        clr     bitcount        ;Reset bitcount to zero.
        bra     getstart        ;Wait for the next start bit.

* The integrity of the received data is determined by cs_test.  The received data is considered
* good if the received checksum correlates with the calculated checksum.
* The checksum location contains the least significant nibble of the sum of all all nibbles received,
* including the meter's checksum nibble.  Because of this, the meter's checksum nibble must be
* subtracted from the calculated checksum prior to comparison.
* The meter's checksum is 1 less than the calculated checksum.  (i.e. a calculated checksum
* of $0f corresponds to a received checksum of $0e).

cs_test lda     firstbyte+$11   ;Load S31 and the meter's checksum into A.
        and     #$0f    ;Remove S31.
        sta     temp1   ;Store the meter's checksum in temp1.

        lda     checksum        ;Load the calculated checksum into A.
        sub     temp1   ;Subtract the meter's checksum nibble.
        deca            ;Subtract 1.
        and     #$0f    ;Knock-off the high nibble.

        ldx     #$18    ;Load the output string offset into X.
        stx     bytecount       ;Bytecount now points to the output string character.

        cmp     temp1   ;See if the calculated checksum matches the meter's checksum.
        beq     ol_test ;If so, procede with decoding.
                        ;If the checksums do not match, send an exclamation point to
                        ;   indicate a checksum error.
        lda     #$21    ;Load the ASCII value for an exclamation point (!).
        sta     string,x        ;Store it as the first character in the output string.
        decx            ;Decrement the string pointer.
        jmp     sendstring      ;Send the exclamation point and restart the program.

* The following code segment, ol_test, determines if the overload message is on the
* display.  To accomplish this, the tens digit is checked for an "L".  If the "L" exists,
* then it is not necessary to process anything else.  If not, then proceed with the
* numeric decoding.  Note: the bottom leg of the "L" character is unique in S9 so it is used
* to check for the entire "L".

ol_test lda     firstbyte+$0a   ;Load S9.
        and     #$0f    ;Knock-off the high nibble.
        cmp     #$04    ;Check for the bottom leg of the "L" character.
        bne     sign    ;Go to the sign routine if the "L" does not exist.
        jmp     overload        ;Go to the overload routine if the "L" exists.

* No overload has been detected so now we can decode the display digits.

* The sign routine checks the acquired data for a minus sign.  If the minus sign is present, the routine
* stores the ASCII value of a minus sign as the first byte of the output string.  If not, the program proceeds
* to decode the ten-thousands digit.

sign    brclr   6,firstbyte+$02,ten_k   ;See if the sign bit is clear.
        lda     #$2d    ;Load the ASCII value for a minus sign.
        jsr     outbyte ;Store a minus sign as the first byte in the output.

* The only digit that can appear in the ten-thousands place is a "1".  The ten_k routine checks for the presence
* of a "1".  Note that this digit will only be filled when the meter is in four digit mode.  If a "1" exists, it is
* stored as the next character in the output string.  If not, the program proceeds to decode the remaining four
* digits.

ten_k   brclr   4,firstbyte+$02,start_thou      ;See if there is a 1 in the ten-thousands place.
        lda     #$31    ;Load the ASCII value for a "1".
        jsr     outbyte ;Store the "1" in the output string.

* Start_thou is the beginning of the decoding routine for the remaining four display digits.

start_thou      stx     bytecount       ;Store the output string pointer.
        ldx     #$0f    ;Load X with the offset of the first S register.
        stx     scount  ;Store it in scount.

thousands       ldx     scount  ;Load the current scount value into X.
        cpx     #$07    ;See if it is the last S register.
        bne     continue        ;If it is the last S register, decode the magnitude.
        bra     magnitude

continue        lda     firstbyte,x     ;Load S14 the 1st time, S12, S10, S8.
        sta     highS   ;Store it in highS.
        decx            ;Decrement scount.
        lda     firstbyte,x     ;Load S13 the 1st time, S11, S9, S7.
        sta     lowS    ;Store it in lowS.
        decx            ;Decrement scount.
        stx     scount  ;Store scount.
        ldx     bytecount       ;Load the current output string offset into X.

display lda     highS   ;Load the high S register.
        asla            ;Move the 4 LSBs to the MSBs.
        asla
        asla
        asla
        sta     temp1   ;Store it in temp1.
        lda     lowS    ;Load the low S register.
        and     #$0f    ;Knock-off the 4MSBs.
        ora     temp1   ;Now we have a byte which represents the digit.
        sta     temp1   ;Store this byte in temp1.

* The dptest routine checks the digit to see if the decimal point is lit.  If it is, dptest stores the
* ASCII value of a decimal point at the current output string location.  If not, it proceeds to decode the
* numerals.

dptest  brclr   6,temp1,numerals        ;Test for a decimal point.
        lda     #$2e    ;The ASCII value for a period (decimal point).
        jsr     outbyte ;Store it in the current output string location.

numerals        lda     temp1   ;Load the digit information into A.
        and     #$bf    ;Get rid of the decimal point bit.
        stx     bytecount       ;Store the output string offset because the X register
                        ;   is needed for numeric decoding.
        clrx
compare cmp     #$af    ;Test for a "0".
        beq     storechar       ;If a "0" is present, go to the storechar routine.
        incx            ;Increment X if a zero is not present.
        cmp     #$09    ;Test for a "1".
        beq     storechar       ;If a "1" is present, go to the storechar routine.
        incx            ;Increment X if a one is not present.
        cmp     #$97    ;Test for a "2".
        beq     storechar       ;If a "2" is present, go to the storechar routine.
        incx            ;Increment X if a two is not present.
        cmp     #$1f    ;Test for a "3".
        beq     storechar       ;If a "3" is present, go to the storechar routine.
        incx            ;Increment X if a three is not present.
        cmp     #$39    ;Test for a "4".
        beq     storechar       ;If a "4" is present, go to the storechar routine.
        incx            ;Increment X if a four is not present.
        cmp     #$3e    ;Test for a "5".
        beq     storechar       ;If a "5" is present, go to the storechar routine.
        incx            ;Increment X if a five is not present.
        cmp     #$be    ;Test for a "6".
        beq     storechar       ;If a "6" is present, go to the storechar routine.
        incx            ;Increment X if a six is not present.
        cmp     #$0b    ;Test for a "7".
        beq     storechar       ;If a "7" is present, go to the storechar routine.
        incx            ;Increment X if a seven is not present.
        cmp     #$bf    ;Test for an "8".
        beq     storechar       ;If an "8" is present, go to the storechar routine.
        incx            ;Increment X if an eight is not present.
        cmp     #$3f    ;Test for a "9".
        bne     thousands       ;If an "9" is present, go to the storechar routine.
                        ;All numeric possibilities have been tested for.  If none of them
                        ;  exist, proceed to decode the next digit.

storechar       txa             ;Transfer the X register to A.  The X register contains
                        ;A value indicative of the numeral on the display.
        add     #$30    ;Add X to the ASCII value for a "0".
        ldx     bytecount       ;Load X with a current output string offset.
        sta     string,x        ;Store the ASCII value of the displayed numeral at the current
                        ;   output string position.
        dec     bytecount       ;Decrement the output string pointer.
        bra     thousands       ;Repeat the numeric decoding process for the remaining digits.

* The overload routine stores the ASCII values for "OL" at the current output string location.

overload        lda     #$4f    ;The ASCII value of an "O"
        jsr     outbyte ;Store it in the current output string location.
        lda     #$4c    ;The ASCII value of an "L"
        jsr     outbyte ;Store it in the current output string location.
        lda     #$20    ;The ASCII value of a space
        jsr     outbyte ;Store it in the current output string location.
        stx     bytecount       ;Store the current output string offset.

* The magnitude routine checks the appropriate S registers for magnitude indicators such as mega, kilo, milli, etc.
* Each magnitude designator has its own ASCII storage routine similar to that of the overload routine.

magnitude       ldx     bytecount
        brset   3,firstbyte+$05,micro   ;Check S4 for a "micro" sign.
        brset   2,firstbyte+$05,nano    ;Check S4 for a "nano" sign.
        brset   1,firstbyte+$05,mega    ;Check S4 for a "mega" sign.
        brset   3,firstbyte+$04,milli   ;Check S3 for a "milli" sign.
        brset   1,firstbyte+$04,kilo    ;Check S3 for a "kilo" sign
        bra     units   ;Go to the units routine if no magnitude designators
                        ;   are detected.

* The outbyte subroutine stores the contents of the accumulator at the current output string location.
* On entry, the X register must contain the offset relative to the string location (see equate table).
* The X register is decremented to prepare for the next output string byte.

outbyte sta     string,x        ;Store the accumulator at string+x.
        decx            ;Decrement X.
        rts             ;Return to caller.

* The following routines store the ASCII values of the magnitude designators.  These routines are entered
* from the magnitude routine.

micro   lda     #$75    ;Load the ASCII values for a "u".
        bra     storemag        ;Go to the magnitude storing routine.
nano    lda     #$6e    ;Load the ASCII values for an "n".
        bra     storemag        ;Go to the magnitude storing routine.
mega    lda     #$4d    ;Load the ASCII values for an "M".
        bra     storemag        ;Go to the magnitude storing routine.
milli   lda     #$6d    ;Load the ASCII values for an "m".
        bra     storemag        ;Go to the magnitude storing routine.
kilo    lda     #$4b    ;Load the ASCII values for a "K".

storemag        jsr     outbyte ;Go to the ASCII byte storing subroutine.

* The units routine checks the appropriate S registers for unit indicators such as farads, volts, ohms, etc.
* Each unit designator has its own ASCII storage routine similar to that of the overload routine.

units   brset   2,firstbyte+$04,farads  ;Check S3 for a farads indicator.
        brset   3,firstbyte+$03,volts   ;Check S2 for a volts indicator.
        brset   2,firstbyte+$03,seconds ;Check S2 for a seconds indicator.
        brset   1,firstbyte+$03,ohms    ;Check S2 for a ohms indicator.
        brset   3,firstbyte+$02,amps    ;Check S1 for a amps indicator.
        brset   2,firstbyte+$02,percent ;Check S1 for a percent indicator.
        brset   1,firstbyte+$02,hertz   ;Check S1 for a hertz indicator.
        bra     comments        ;Jump to the comments routine if no units indicators were
                        ;   detected.

* The following routines store the ASCII values of the units designators.  These routines are entered
* from the unit routine.

farads  lda     #$46    ;The ASCII value of "F".
        bra     storeunit
volts   lda     #$56    ;The ASCII value of "V".
        bra     storeunit
seconds lda     #$73    ;The ASCII value of "s".
        bra     storeunit
ohms    lda     #$6f    ;The ASCII value of "o".
        jsr     outbyte
        lda     #$68    ;The ASCII value of "h".
        jsr     outbyte
        lda     #$6d    ;The ASCII value of "m".
        jsr     outbyte
        bra     seconds ;Pluralize "ohm" using the "s" from the seconds routine to
                        ;   save ROM space.
amps    lda     #$41    ;The ASCII value of "A".
        bra     storeunit
percent lda     #$25    ;The ASCII value of "%".
        bra     storeunit
hertz   lda     #$48    ;The ASCII value of "H".
        jsr     outbyte
        lda     #$7a    ;The ASCII value of "z".

storeunit       jsr     outbyte

* The comments routine detects the "AC" and "DC" annunciators.

comments        brset   3,firstbyte+$01,alternate       ;Check S0 for the alternating current annunciator.
        brset   2,firstbyte+$01,direct  ;Check S0 for the direct current annunciator.
        bra     sendstring      ;Jump to the serial output routine if neither of these exist.

alternate       lda     #$41    ;The ASCII value of "A".
        jsr     outbyte ;Store the "A" at the current output string location.
        bra     current ;Jump to the routine to store a "C".
direct  lda     #$44    ;The ASCII value of "D".
        jsr     outbyte ;Store the "D" at the current output string location.
current lda     #$43    ;The ASCII value of "C".
        jsr     outbyte ;Store the "D" at the current output string location.

* The sendstring routine terminates the output string with a <CR> and sends the output string, in serial format at
* 9600bps 8N1, to porta bit 0 (where the MAX 232A serial driver is connected).

sendstring      lda     #$0d    ;The ASCII value of a <CR>.
        sta     string,x        ;Store the <CR> at the current output string location.  The <CR>
                        ;   provides a carriage return and serves as the output string
                        ;   terminator.

        lda     #$18    ;Load the offset of the first character in the output string.
        sta     bytecount       ;Store it in bytecount.

serstart        ldx     #$08    ;The number of bits in a byte.
        stx     bitcount        ;Store it in bitcount.

startbit        clr     porta   ;Send a start bit.
        ldx     #$20    ;Load X as a countdown register with the serial bit time.
startdly        decx
        bne     startdly        ;Loop until the serial bit time has expired.

        ldx     bytecount       ;Load X with the current output string offset.
        lda     string,x        ;Load A with the current output string character.

sendbit sta     porta   ;The bit of interest is stored at porta bit 0.
        ldx     #$20    ;Load X as a countdown register with the serial bit time.
bitdelay        decx
        bne     bitdelay        ;Loop until the serial bit time has expired.
        rora            ;Rotate the data byte to get the next bit if interest into
                        ;   the LSB of A.
        dec     bitcount        ;Decrement bitcount.
        bne     sendbit ;Repeat the bit shifting if eight data bits have not yet been sent.

stopbit bset    0,porta ;Send a stop bit.

bytedelay       decx            ;Delay before next character. (X is $00 on entry)
        bne     bytedelay       ;Loop until the interbyte delay has expired.
        ldx     bytecount       ;Load X with the current output string offset.
        lda     string,x        ;Load the current output string character into A.
        cmp     #$0d    ;See if it is the <CR> termination character.
        beq     end     ;If it is the termination character, exit the sendserial routine.
        dec     bytecount       ;Update the current output string offset.
        bra     serstart        ;Send the next byte in the output string.

end     jmp     reset   ;Go to the beginning of the program and start the entire
                        ;   process again.
