*************************************
*     RS-232 MuxBox sourcecode      *
*                                   *
*        Author: Derek Matsunaga    *
*      Revision: 1.0                *
* Revision date: November 8, 1998   *
*     Processor: 68HC705J1A         *
*                                   *
*************************************

;    Notes: Labels, mnemonics, operands, and comments are separated by tabs.  Set your editor
;           accordingly to view the sourcecode in readable columns.  The sourcecode is best
;           viewed using tab stops and a fixed space typeface.

;           Please send comments and feedback to derek700@aol.com

PORTA           EQU     $00
PORTB           EQU     $01
DDRA            EQU     $04
DDRB            EQU     $05
TCR             EQU     $08
TDR             EQU     $09
PDRA            EQU     $10
PDRB            EQU     $11
ISCR            EQU     $0A
errbit          equ     $3
heartpin        equ     $4
heartinterval   equ     !28
datain_time     equ     !10
loop_pin        equ     $1
TX_pin          equ     $0
datain_pin      equ     $5
button_pin      equ     $2
;statebits:
parse           equ     $0
outchar         equ     $1
debounce1       equ     $2
debounce2       equ     $3
datain_state    equ     $4
datain_delay    equ     $5

        org     $c0
statebyte       rmb     1
datain_count_l  rmb     1
datain_count_h  rmb     1
debounce_count  rmb     1
heartcount_l    rmb     1
heartcount_h    rmb     1
bitcount        rmb     1
sendcount       rmb     1
selection       rmb     1
inbuf_ptr       rmb     1
inbuf           rmb     4
outbuf_ptr      rmb     1
outbuf          rmb     5

;MOR        
	ORG     $7F1
        DB      $00

;Vector table:              
	ORG     $7F8
	DW      RESET           ;timer (never used)
        DW      get232          ;IRQ (232 in)
	DW      RESET           ;SWI (never used)
	DW      RESET           ;power on reset


	ORG     $0300
reset:          rsp                     ;2 Reset stack pointer to top of RAM.
                sei                     ;2 Disable interrupts.
                lda     #$3b
                sta     ddrb            ;Configure port B
                lda     #$FF
                sta     ddra            ;Configure port A
                lda     #$fb
                sta     pdrb            ;Pulldown for PB2
                bset    TX_pin,PORTB    ;Initialize 232 TX.
                clr     outbuf_ptr
                clr     inbuf_ptr
                clr     statebyte
                clr     selection
                clr     debounce_count
                lda     #heartinterval
                sta     heartcount_h
                cli                     ;Enable interrupts
main:           bset    loop_pin,PORTB
                bclr    loop_pin,PORTB
                jsr     heartbeat
                jsr     send232
                jsr     check_button
                jsr     parser
                jsr     write_mux
                jsr     data_in_LED
                bra     main

data_in_LED:    brset   datain_state,statebyte,reset_count
                brclr   datain_delay,statebyte,data_in_out
                dec     datain_count_l
                bne     data_in_out
                dec     datain_count_h
                bne     data_in_out
                bset    datain_pin,PORTB        ;Extinguish LED.
                bclr    datain_delay,statebyte
data_in_out:    rts

reset_count:    clr     datain_count_l
                lda     #datain_time 
                sta     datain_count_h
                bclr    datain_state,statebyte
                bset    datain_delay,statebyte
                bclr    datain_pin,PORTB        ;Light LED
                rts

write_mux:      lda     porta
                and     #$f8
                ora     selection
                sta     porta
                rts

heartbeat:      dec     heartcount_l
                bne     heartout
                dec     heartcount_h
                bne     heartout
                brset   heartpin,portb,heartoff
                bset    heartpin,portb
                bra     heartreset
heartoff:       bclr    heartpin,portb
heartreset:     lda     #heartinterval
                sta     heartcount_h
heartout:       rts

parser:         brclr   parse,statebyte,parse_out
                bclr    parse,statebyte
                lda     inbuf
                cmp     #$0d                    ;Check for <cr> only.
                beq     no_err
                cmp     #$0a                    ;Check for <lf>.
                beq     no_err
                cmp     #'S'
                beq     cmd_set
                cmp     #'s'
                beq     cmd_set
                cmp     #'?'
                beq     cmd_inq
                cmp     #'R'
                beq     cmd_reset
                cmp     #'r'
                beq     cmd_reset
                bra     error
cmd_set:        lda     inbuf+1
                cmp     #'0'
                blo     error
                cmp     #'7'
                bhi     error
                sub     #'0'
                and     #$07
                sta     selection
                bra     no_err
cmd_reset:      jmp     RESET
cmd_inq:        lda     selection
                add     #'0'
                sta     outbuf+2
                lda     #'='
                sta     outbuf+1
                lda     #'S'
                sta     outbuf
                lda     #$0d
                sta     outbuf+3
                lda     #$0a
                sta     outbuf+4
                clr     outbuf+5                ;Termination character.
                clr     outbuf_ptr
                bset    outchar,statebyte
no_err:         bset    errbit,portb
                clr     inbuf_ptr       ;  Reset inbuf.
parse_out:      rts

error:          bclr    errbit,portb
                clr     inbuf_ptr       ;  Reset inbuf.
                cli                     ;  Re-enable interrupts.
                rts

;get232 is an ISR triggered by a 232 start bit at \IRQ.
get232:         sei                     ;2 Disable interrupts.
                lda     #$08            ;2
                sta     bitcount        ;4
                lda     inbuf_ptr       ;3
                and     #$03            ;2
                tax                     ;2
                bsr     wait_halfbit    ;6 11.5us to here. Need 40.5us more
                                        ;  52us to here.

getbit:         bsr     wait_1bit       ;6 Getbit loop = 14us + delay.

                bih     shift_one       ;3
shift_zero:     clc                     ;2
                ror     inbuf,x         ;6
                bra     next_bit        ;3 (14) 7us
shift_one:      sec                     ;2
                ror     inbuf,x         ;6
                bra     next_bit        ;3 (14) Keep the timing symmetrical.

next_bit:       dec     bitcount        ;5
                bne     getbit          ;3

                lda     inbuf,x         ;4
                cmp     #$0a            ;2   Check for <LF>.
                beq     lf_in
                cmp     #$0d            ;2   Check for <CR>.
                bne     no_cr           ;3
                bset    parse,statebyte ;5
no_cr:          inc     inbuf_ptr       ;5
lf_in:          bsr     wait_stop_bit   ;6   Wait until stop bit.
                bset    1,ISCR          ;5
                bset    datain_state,statebyte  ;5   Indicate data_in.
                cli                     ;2   Re-enable interrupts.
                rti                     ;6

wait_stop_bit:  lda     #!18            ;2   60us delay.
                bra     bit_delay;      ;3
wait_halfbit:   lda     #!11            ;2   40.5us delay.
                bra     bit_delay       ;3
wait_1bit:      lda     #!28            ;2   Set to 90us to add to 14us/bit.
bit_delay:      nop                     ;2
                nop                     ;2
bit_loop:       deca                    ;3
                bne     bit_loop        ;3
                rts                     ;6

send232:        brclr   outchar,statebyte,send232_out
                ldx     outbuf_ptr

                lda     outbuf,x        ;4
                beq     send232_done    ;3

                clc
                bsr     send_bit        ;6  Send a start bit.
                lda     #$08            ;2
                sta     sendcount       ;4  19 cycles to here + send_bit.

shift_out:      ror     outbuf,x        ;6  Shift_out = 20 cycles + send_bit.
                bsr     send_bit        ;6
                dec     sendcount       ;5
                bne     shift_out       ;3

                sec                     ;2
                bsr     send_bit        ;6  Send the top bit.
                
                clr     outbuf,x        ;5
                inc     outbuf_ptr      ;5

send232_out:    rts
send232_done:   bclr    outchar,statebyte
                clr     outbuf_ptr
                clr     outbuf
                rts

;The carry bit contains the bit to be sent.  A is destroyed.
send_bit:       bcc     send_zero       ;3
send_one:       bset    TX_pin,PORTB    ;5
                bra     send_delay      ;3
send_zero:      bclr    TX_pin,PORTB    ;5
                bra     send_delay      ;3
send_delay:     lda     #!28            ;2  13 cycles including here.
send_loop:      deca                    ;3
                bne     send_loop       ;3
                rts                     ;6

check_button:   brset   debounce1,statebyte,debouncer1
                brset   debounce2,statebyte,debouncer2
                brset   button_pin,PORTB,pressed
                rts

debouncer1:     dec     debounce_count
                bne     button_out
                bclr    debounce1,statebyte
                brclr   button_pin,PORTB,button_out
                lda     selection
                inca
                and     #$07
                sta     selection
                bset    debounce2,statebyte
                bra     button_out

debouncer2:     brset   button_pin,PORTB,button_out
                dec     debounce_count
                bne     button_out
                brset   button_pin,PORTB,button_out
                bclr    debounce2,statebyte
                bra     button_out

pressed:        bset    debounce1,statebyte
button_out:     rts
